// ================= CONFIG & STATE =================
const DEFAULT_CONFIG = {
    // 全局/背景
    blur: 10,
    brightness: 20,
    bgType: 'image',
    bgUrl: '',
    
    // Logo
    showLogo: true,
    logoText: 'GlassTab',
    logoSize: 4.5, 

    // 时间
    showTime: true,
    timeSize: 4.5,
    timeWeight: '700',
    timeFont: 'sans-serif',
    
    // 日期
    dateSize: 1.2,
    dateWeight: '400',
    dateFont: 'sans-serif',

    // 图标布局
    iconSize: 64,
    iconRadius: 'rounded-2xl',
    gridCols: 6,
    openNewTab: false,
    // [新增] 图标玻璃背景的模糊度 (默认20px)
    appGlassBlur: 20, 
    // [新增] 内部图标的投影强度 (none, sm, md, lg)
    innerIconShadow: 'md',
    // [新增] 图片类图标的缩放比例 (30-100, 默认 60 以匹配字体大小)
    iconImgScale: 60,
    
    // 图标文字
    textSize: 0.9,
    textWeight: '500',
    textFont: 'sans-serif',
    textShadow: 'sm',
    
    // 搜索
    engine: 'google',
    customEngineUrl: '',
    searchNewTab: false
};

const defaultApps = [
    { id: 1, name: "书签管理", url: "chrome://bookmarks/", icon: "fas fa-star", type: "font", color: "#FBC02D" },
    { id: 2, name: "扩展程序", url: "chrome://extensions/", icon: "fas fa-puzzle-piece", type: "font", color: "#757575" },
    { id: 3, name: "Google", url: "https://google.com", icon: "fab fa-google", type: "font", color: "#4285F4" },
    { id: 4, name: "YouTube", url: "https://youtube.com", icon: "fab fa-youtube", type: "font", color: "#FF0000" },
    { id: 5, name: "Bilibili", url: "https://bilibili.com", icon: "fab fa-bilibili", type: "font", color: "#00A1D6" },
    { id: 6, name: "Steam", url: "steam://store/", icon: "fab fa-steam", type: "font", color: "#171a21" },
];

function loadState() {
    try {
        const localApps = JSON.parse(localStorage.getItem('glassTab_apps'));
        const localConfig = JSON.parse(localStorage.getItem('glassTab_config'));
        return {
            apps: localApps || defaultApps,
            config: Object.assign({}, DEFAULT_CONFIG, localConfig || {})
        };
    } catch (e) {
        console.error("Config load failed, using default:", e);
        return { apps: defaultApps, config: DEFAULT_CONFIG };
    }
}

let state = loadState();

const searchEngines = {
    google: { url: "https://www.google.com/search?q=", icon: "fab fa-google" },
    bing: { url: "https://www.bing.com/search?q=", icon: "fab fa-microsoft" },
    ddg: { url: "https://duckduckgo.com/?q=", icon: "fas fa-duck" }
};

let currentEditId = null;

// ================= INIT =================
window.onload = () => {
    // 页面加载完成后显示背景，修复背景黑屏问题
    const bgLayer = document.getElementById('bg-layer');
    if (bgLayer) setTimeout(() => bgLayer.classList.add('loaded'), 100);
};

document.addEventListener('DOMContentLoaded', () => {
    if (!state.config.gridCols) state.config.gridCols = 6;
    state.config = Object.assign({}, DEFAULT_CONFIG, state.config);
    
    initClock();
    renderGrid();
    applyConfig();
    initSearch();
    initSortable();
    initContextMenus();
    initSettingsListeners();
    initIconEditorListeners();
});

function saveState() {
    localStorage.setItem('glassTab_apps', JSON.stringify(state.apps));
    localStorage.setItem('glassTab_config', JSON.stringify(state.config));
    applyConfig(); 
}

// ================= CLOCK =================
function initClock() {
    const timeEl = document.getElementById('time-display');
    const dateEl = document.getElementById('date-display');
    const container = document.getElementById('clock-container');
    
    function update() {
        if (!state.config.showTime) {
            container.classList.add('hidden');
            return;
        }
        container.classList.remove('hidden');
        
        const now = new Date();
        const days = ['星期日', '星期一', '星期二', '星期三', '星期四', '星期五', '星期六'];
        
        const h = String(now.getHours()).padStart(2, '0');
        const m = String(now.getMinutes()).padStart(2, '0');
        const s = String(now.getSeconds()).padStart(2, '0');
        timeEl.textContent = `${h}:${m}:${s}`;

        const y = now.getFullYear();
        const mm = String(now.getMonth() + 1).padStart(2, '0');
        const dd = String(now.getDate()).padStart(2, '0');
        dateEl.textContent = `${y}/${mm}/${dd} ${days[now.getDay()]}`;
    }
    setInterval(update, 1000);
    update();
}

// ================= RENDER GRID(核心修改) =================
function renderGrid() {
    const grid = document.getElementById('app-grid');
    if (!grid) return;
    grid.innerHTML = '';
    
    let cols = parseInt(state.config.gridCols);
    if (isNaN(cols) || cols < 1) cols = 6;
    grid.style.gridTemplateColumns = `repeat(${cols}, minmax(0, 1fr))`;
    
    const getTextShadow = (s) => {
        if (s === 'none') return 'none';
        return '0 1px 2px rgba(0,0,0,0.8)';
    };

    const getIconFilter = (s) => {
        if (!s || s === 'none') return 'none';
        if (s === 'sm') return 'drop-shadow(0 2px 2px rgba(0,0,0,0.5))';
        if (s === 'md') return 'drop-shadow(0 4px 6px rgba(0,0,0,0.6))';
        if (s === 'lg') return 'drop-shadow(0 6px 12px rgba(0,0,0,0.8))';
        return 'none';
    };
    const iconFilterStyle = getIconFilter(state.config.innerIconShadow);

    // [获取配置] 背景显隐、模糊度、图片缩放
    const showBg = state.config.appGlassBg !== undefined ? state.config.appGlassBg : true;
    const glassBlurVal = (state.config.appGlassBlur !== undefined) ? state.config.appGlassBlur : 20;
    const imgScale = (state.config.iconImgScale !== undefined) ? state.config.iconImgScale : 60;

    if (!state.apps || state.apps.length === 0) return;

    state.apps.forEach(app => {
        const el = document.createElement('div');
        el.className = 'app-item flex flex-col items-center gap-2 group cursor-pointer relative select-none';
        el.setAttribute('data-id', app.id);
        
        const iconBox = document.createElement('div');
        const radius = state.config.iconRadius || 'rounded-2xl';
        
        // [逻辑修改] 根据 showBg 决定是否添加 glass-panel 类和阴影
        // 如果不显示背景，只保留基本布局属性，去掉 glass-panel 和 shadow
        let baseClass = `${radius} flex items-center justify-center transition-all duration-300`;
        if (showBg) {
            baseClass += ` glass-panel shadow-lg`;
        }
        iconBox.className = baseClass;
        
        const size = parseInt(state.config.iconSize) || 64;
        iconBox.style.width = `${size}px`;
        iconBox.style.height = `${size}px`;
        // 字体图标的大小基准（通常是容器的一半）
        iconBox.style.fontSize = `${size * 0.5}px`;
        
        // [逻辑修改] 只有开启背景时，才应用 backdrop-filter
        if (showBg) {
            iconBox.style.backdropFilter = `blur(${glassBlurVal}px)`;
            iconBox.style.webkitBackdropFilter = `blur(${glassBlurVal}px)`;
            // 如果开启背景，悬浮时稍微放大一点容器
            // (可以通过 CSS group-hover 实现，也可以在这里留白)
        } else {
            // 如果没背景，清除滤镜
            iconBox.style.backdropFilter = 'none';
            iconBox.style.webkitBackdropFilter = 'none';
            iconBox.style.background = 'transparent'; // 确保完全透明
            iconBox.style.border = 'none'; // 去掉边框
            iconBox.style.boxShadow = 'none'; // 去掉盒子阴影
        }

        if (app.type === 'font') {
            iconBox.innerHTML = `<i class="${app.icon}" style="color: ${app.color || '#ffffff'}; filter: ${iconFilterStyle}"></i>`;
        } else {
            // [核心修改] 图片不再使用 w-full h-full，而是使用 config 中的百分比
            // object-fit: contain 保证图片不比例失调
            iconBox.innerHTML = `<img src="${app.icon}" class="${radius} object-contain" style="width: ${imgScale}%; height: ${imgScale}%; filter: ${iconFilterStyle}">`;
        }

        const label = document.createElement('span');
        label.className = 'truncate w-full text-center px-1 transition-all';
        label.textContent = app.name;
        label.style.fontSize = `${state.config.textSize || 0.9}rem`;
        label.style.fontWeight = state.config.textWeight || '500';
        label.style.fontFamily = state.config.textFont || 'sans-serif';
        label.style.textShadow = getTextShadow(state.config.textShadow);

        el.appendChild(iconBox);
        el.appendChild(label);

        el.addEventListener('click', () => {
            if (!grid.classList.contains('sorting')) openUrl(app.url);
        });

        grid.appendChild(el);
    });
}

//
function openUrl(url) {
    if (!url) return;

    // 1. 智能协议检测
    // 正则匹配：以字母开头，后接 :// 的格式 (例如 chrome://, steam://, mailto://)
    const hasProtocol = /^[a-zA-Z0-9\-\.]+:\/\//.test(url);

    // 2. 如果没有检测到任何协议，才默认补全 https://
    if (!hasProtocol) {
        url = 'https://' + url;
    }

    // 3. 针对浏览器内部链接 (chrome://, edge://, about:) 的特殊处理
    // 普通网页无法跳转这些链接，必须使用 Extension API
    if (url.startsWith('chrome://') || url.startsWith('edge://') || url.startsWith('about:')) {
        if (typeof chrome !== 'undefined' && chrome.tabs) {
            // 使用 chrome.tabs API 进行跳转
            if (state.config.openNewTab) {
                chrome.tabs.create({ url: url });
            } else {
                chrome.tabs.update({ url: url });
            }
        } else {
            // 如果在普通网页环境调试（非扩展环境），提示用户
            console.warn("Chrome API not found. Cannot navigate to internal pages in standard web environment.");
            alert("系统提示：\n浏览器内部页面 (chrome://) 仅在插件正式安装后才能跳转。");
        }
        return; // 结束执行
    }

    // 4. 处理常规链接和外部协议 (https, steam, magnet 等)
    // steam:// 这类协议可以直接通过 window.open 或 location 触发
    if (state.config.openNewTab) {
        window.open(url, '_blank');
    } else {
        window.location.href = url;
    }
}

// ================= SORTABLE =================
function initSortable() {
    const grid = document.getElementById('app-grid');
    if (typeof Sortable !== 'undefined' && grid) {
        new Sortable(grid, {
            animation: 150,
            onStart: () => grid.classList.add('sorting'),
            onEnd: () => {
                grid.classList.remove('sorting');
                const newOrderIds = Array.from(grid.querySelectorAll('.app-item[data-id]')).map(el => parseInt(el.getAttribute('data-id')));
                const newApps = [];
                newOrderIds.forEach(id => {
                    const app = state.apps.find(a => a.id === id);
                    if(app) newApps.push(app);
                });
                state.apps = newApps;
                saveState();
            }
        });
    }
}

// ================= SEARCH =================
function initSearch() {
    const btn = document.getElementById('search-engine-btn');
    const dropdown = document.getElementById('engine-dropdown');
    const input = document.getElementById('search-input');
    const icon = document.getElementById('current-engine-icon');
    const submit = document.getElementById('search-submit');

    const updateIcon = () => {
        if (!state.config.engine) state.config.engine = 'google';
        const eng = searchEngines[state.config.engine] || searchEngines.google;
        icon.className = eng.icon + " text-xl";
    };
    updateIcon();

    btn.addEventListener('click', (e) => {
        e.stopPropagation();
        dropdown.classList.toggle('hidden');
        dropdown.classList.toggle('flex');
    });

    document.querySelectorAll('.engine-option').forEach(opt => {
        opt.addEventListener('click', () => {
            const eng = opt.getAttribute('data-engine');
            state.config.engine = eng;
            updateIcon();
            dropdown.classList.add('hidden');
            dropdown.classList.remove('flex');
            saveState();
        });
    });

    window.addEventListener('click', () => {
        if(dropdown) {
            dropdown.classList.add('hidden');
            dropdown.classList.remove('flex');
        }
    });

    function performSearch() {
        const query = input.value;
        if (!query) return;
        
        let url = '';
        const currentEng = searchEngines[state.config.engine] || searchEngines.google;
        
        if (state.config.customEngineUrl && state.config.customEngineUrl.trim() !== '') {
             url = state.config.customEngineUrl + encodeURIComponent(query);
        } else {
             url = currentEng.url + encodeURIComponent(query);
        }

        if (state.config.searchNewTab) {
            window.open(url, '_blank');
        } else {
            window.location.href = url;
        }
    }

    input.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') performSearch();
    });
    submit.addEventListener('click', performSearch);
}

// ================= CONTEXT MENU =================
const ctxMenu = document.getElementById('context-menu');
let ctxTargetId = null;

function showContextMenu(e) {
    e.preventDefault();
    const appItem = e.target.closest('.app-item');
    const btnEdit = document.getElementById('ctx-edit');
    const btnDelete = document.getElementById('ctx-delete');
    const btnAdd = document.getElementById('ctx-add');
    const separator = document.querySelector('.menu-separator');

    if (appItem) {
        ctxTargetId = parseInt(appItem.getAttribute('data-id'));
        btnEdit.classList.remove('hidden');
        btnDelete.classList.remove('hidden');
        if(separator) separator.classList.remove('hidden');
        btnAdd.classList.add('hidden');
    } else {
        ctxTargetId = null;
        btnEdit.classList.add('hidden');
        btnDelete.classList.add('hidden');
        if(separator) separator.classList.add('hidden');
        btnAdd.classList.remove('hidden');
    }

    let x = e.clientX;
    let y = e.clientY;
    const menuWidth = 128; const menuHeight = 110;
    if (x + menuWidth > window.innerWidth) x -= menuWidth;
    if (y + menuHeight > window.innerHeight) y -= menuHeight;
    ctxMenu.style.left = `${x}px`;
    ctxMenu.style.top = `${y}px`;
    ctxMenu.classList.remove('hidden');
    ctxMenu.classList.add('flex');
}

function initContextMenus() {
    document.addEventListener('contextmenu', (e) => {
        if (e.target.tagName !== 'INPUT' && e.target.tagName !== 'TEXTAREA') {
            showContextMenu(e);
        }
    });
    document.addEventListener('click', (e) => {
        if (ctxMenu && !ctxMenu.contains(e.target)) {
            ctxMenu.classList.add('hidden');
            ctxMenu.classList.remove('flex');
        }
    });
    // === 替换为这段新代码 ===
    document.getElementById('ctx-delete').addEventListener('click', () => {
        ctxMenu.classList.add('hidden'); // 关闭右键菜单
        showConfirmModal();              // 呼出自定义玻璃弹窗
    });
    document.getElementById('ctx-edit').addEventListener('click', () => {
        openIconModal(ctxTargetId);
        ctxMenu.classList.add('hidden');
    });
    document.getElementById('ctx-add').addEventListener('click', () => {
        openIconModal();
        ctxMenu.classList.add('hidden');
    });
}

// ================= SETTINGS LOGIC =================
function toggleSettings(show) {
    const modal = document.getElementById('settings-modal');
    const overlay = document.getElementById('settings-overlay');
    if (show) {
        modal.classList.add('active');
        overlay.classList.add('active');
        loadSettingsToUI();
    } else {
        modal.classList.remove('active');
        overlay.classList.remove('active');
    }
}

function initSettingsListeners() {
    document.getElementById('settings-btn').addEventListener('click', () => toggleSettings(true));
    document.getElementById('settings-overlay').addEventListener('click', () => toggleSettings(false));
    
    const bind = (id, key, suffix = '', needsRender = false) => {
        const el = document.getElementById(id);
        const valDisplay = document.getElementById(id.replace('conf-', 'val-')); 
        if(!el) return;
        el.addEventListener('input', (e) => {
            const val = e.target.type === 'checkbox' ? e.target.checked : e.target.value;
            state.config[key] = val;
            if(valDisplay) valDisplay.textContent = val + suffix;
            applyConfig();
            if(needsRender) renderGrid();
        });
        el.addEventListener('change', saveState);
    };

    bind('conf-blur', 'blur', 'px');
    bind('conf-brightness', 'brightness', '%');
    bind('conf-show-logo', 'showLogo');
    bind('conf-logo-text', 'logoText');
    bind('conf-logo-size', 'logoSize', 'rem');
    bind('conf-show-time', 'showTime');
    bind('conf-time-size', 'timeSize', 'rem');
    bind('conf-time-weight', 'timeWeight');
    bind('conf-time-font', 'timeFont'); 
    bind('conf-date-size', 'dateSize', 'rem');
    bind('conf-date-weight', 'dateWeight');
    bind('conf-date-font', 'dateFont');
    bind('conf-icon-size', 'iconSize', 'px', true);
    bind('conf-grid-cols', 'gridCols', '', true);
    bind('conf-icon-radius', 'iconRadius', '', true);
    bind('conf-open-newtab', 'openNewTab');
    bind('conf-text-size', 'textSize', 'rem', true);
    bind('conf-text-weight', 'textWeight', '', true);
    bind('conf-text-font', 'textFont', '', true);
    bind('conf-text-shadow', 'textShadow', '', true);
    bind('conf-search-newtab', 'searchNewTab');
    bind('conf-custom-engine', 'customEngineUrl');
    bind('conf-app-blur', 'appGlassBlur', 'px', true); // 图标背景模糊
    bind('conf-icon-inner-shadow', 'innerIconShadow', '', true); // 内部图标投影
    bind('conf-app-bg', 'appGlassBg', '', true); // 开启/关闭背景
    bind('conf-icon-img-scale', 'iconImgScale', '%', true); // 图片缩放

    const bgUrlInput = document.getElementById('bg-url-input');
    if(bgUrlInput) {
        bgUrlInput.addEventListener('change', (e) => {
            const url = e.target.value;
            state.config.bgUrl = url;
            state.config.bgType = (url.endsWith('.mp4') || url.endsWith('.webm')) ? 'video' : 'image';
            saveState();
        });
    }
    document.getElementById('bg-upload').addEventListener('change', (e) => {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = (evt) => {
                state.config.bgUrl = evt.target.result;
                state.config.bgType = file.type.startsWith('video') ? 'video' : 'image';
                saveState();
            };
            reader.readAsDataURL(file);
        }
    });
    document.getElementById('bg-use-bing').addEventListener('click', () => {
         state.config.bgUrl = 'https://bing.biturl.top/?resolution=1920&format=image&index=0&mkt=zh-CN';
         state.config.bgType = 'image';
         saveState();
    });
    document.getElementById('bg-use-default').addEventListener('click', () => {
        state.config.bgUrl = '';
        state.config.bgType = 'image';
        saveState();
    });
    document.getElementById('btn-backup').addEventListener('click', () => {
        const dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(state));
        const a = document.createElement('a');
        a.href = dataStr; a.download = "glasstab_backup.json"; a.click();
    });
    document.getElementById('file-restore').addEventListener('change', (e) => {
        const file = e.target.files[0];
        if (!file) return;
        const reader = new FileReader();
        reader.onload = (e) => {
            try {
                const data = JSON.parse(e.target.result);
                if(data.apps) {
                    state = data;
                    state.config = Object.assign({}, DEFAULT_CONFIG, state.config);
                    saveState();
                    renderGrid();
                    alert('恢复成功');
                }
            } catch(err) { alert('文件错误'); }
        };
        reader.readAsText(file);
    });
}

function loadSettingsToUI() {
    const setVal = (id, val) => { const el = document.getElementById(id); if(el && val !== undefined) el.value = val; };
    const setCheck = (id, val) => { const el = document.getElementById(id); if(el && val !== undefined) el.checked = val; };
    const setText = (id, val, suffix='') => { const el = document.getElementById(id); if(el) el.textContent = val + suffix; };

    setVal('conf-blur', state.config.blur); setText('val-blur', state.config.blur, 'px');
    setVal('conf-brightness', state.config.brightness); setText('val-brightness', state.config.brightness, '%');
    setCheck('conf-show-logo', state.config.showLogo);
    setVal('conf-logo-text', state.config.logoText);
    setVal('conf-logo-size', state.config.logoSize); setText('val-logo-size', state.config.logoSize, 'rem');
    setCheck('conf-show-time', state.config.showTime);
    setVal('conf-time-size', state.config.timeSize); setText('val-time-size', state.config.timeSize, 'rem');
    setVal('conf-time-weight', state.config.timeWeight);
    setVal('conf-time-font', state.config.timeFont);
    setVal('conf-date-size', state.config.dateSize); setText('val-date-size', state.config.dateSize, 'rem');
    setVal('conf-date-weight', state.config.dateWeight);
    setVal('conf-date-font', state.config.dateFont);
    setVal('bg-url-input', (state.config.bgUrl && state.config.bgUrl.startsWith('data:')) ? '' : state.config.bgUrl);
    setVal('conf-icon-size', state.config.iconSize); setText('val-icon-size', state.config.iconSize, 'px');
    setVal('conf-grid-cols', state.config.gridCols); setText('val-grid-cols', state.config.gridCols, '');
    setVal('conf-icon-radius', state.config.iconRadius);
    setCheck('conf-open-newtab', state.config.openNewTab);
    setVal('conf-text-size', state.config.textSize); setText('val-text-size', state.config.textSize, 'rem');
    setVal('conf-text-weight', state.config.textWeight);
    setVal('conf-text-font', state.config.textFont);
    setVal('conf-text-shadow', state.config.textShadow);
    setCheck('conf-search-newtab', state.config.searchNewTab);
    setVal('conf-custom-engine', state.config.customEngineUrl);
    setVal('conf-app-blur', state.config.appGlassBlur !== undefined ? state.config.appGlassBlur : 20);
    setText('val-app-blur', state.config.appGlassBlur !== undefined ? state.config.appGlassBlur : 20, 'px');
    
    setCheck('conf-app-bg', state.config.appGlassBg !== undefined ? state.config.appGlassBg : true);
    setVal('conf-icon-img-scale', state.config.iconImgScale !== undefined ? state.config.iconImgScale : 60);
    setText('val-icon-img-scale', state.config.iconImgScale !== undefined ? state.config.iconImgScale : 60, '%');

    setVal('conf-icon-inner-shadow', state.config.innerIconShadow || 'md');
}

function applyConfig() {
    const bgLayer = document.getElementById('bg-layer');
    const bgOverlay = document.getElementById('bg-overlay');
    const video = document.getElementById('bg-video');
    
    if (state.config.bgType === 'video' && state.config.bgUrl) {
        bgLayer.style.backgroundImage = 'none';
        video.src = state.config.bgUrl;
        video.classList.remove('hidden');
        video.play().catch(e=>{});
    } else {
        video.classList.add('hidden');
        video.pause();
        if (state.config.bgUrl) {
            bgLayer.style.backgroundImage = `url('${state.config.bgUrl}')`;
        } else {
            bgLayer.style.backgroundImage = `url('https://images.unsplash.com/photo-1477346611705-65d1883cee1e?q=80&w=1920&auto=format&fit=crop')`; 
        }
    }
    
    bgLayer.style.filter = `blur(${state.config.blur || 0}px)`;
    bgOverlay.style.backgroundColor = `rgba(0,0,0, ${(state.config.brightness || 20) / 100})`;
    
    const logoArea = document.getElementById('logo-area');
    const logoText = document.getElementById('logo-text');
    const separator = document.querySelector('.separator');
    const timeDisplay = document.getElementById('time-display');
    const dateDisplay = document.getElementById('date-display');
    const timeContainer = document.getElementById('clock-container');

    if(state.config.showLogo) {
        logoArea.classList.remove('hidden');
        if(separator) separator.classList.remove('hidden'); 
        logoText.textContent = state.config.logoText || 'GlassTab';
        logoText.style.fontSize = `${state.config.logoSize || 4.5}rem`;
    } else {
        logoArea.classList.add('hidden');
        if(separator) separator.classList.add('hidden');
    }

    if (state.config.showTime) {
        timeContainer.classList.remove('hidden');
        timeDisplay.style.fontSize = `${state.config.timeSize || 4.5}rem`;
        timeDisplay.style.fontWeight = state.config.timeWeight || '700';
        timeDisplay.style.fontFamily = state.config.timeFont || 'sans-serif';
        
        dateDisplay.style.fontSize = `${state.config.dateSize || 1.2}rem`;
        dateDisplay.style.fontWeight = state.config.dateWeight || '400';
        dateDisplay.style.fontFamily = state.config.dateFont || state.config.timeFont || 'sans-serif';
    } else {
        timeContainer.classList.add('hidden');
    }
}

// ================= ICON EDITOR LOGIC =================
function toggleIconModal(show) {
    const wrapper = document.getElementById('icon-modal-wrapper');
    const content = document.getElementById('icon-modal-content');
    if (show) {
        wrapper.classList.add('active');
        content.classList.remove('scale-95');
        content.classList.add('scale-100');
    } else {
        wrapper.classList.remove('active');
        content.classList.remove('scale-100');
        content.classList.add('scale-95');
    }
}

function initIconEditorListeners() {
    const typeFontBtn = document.getElementById('type-font');
    const typeImgBtn = document.getElementById('type-img');
    const colorPicker = document.getElementById('icon-color-picker');
    const colorHex = document.getElementById('icon-color-hex');
    const previewIcon = document.getElementById('preview-icon');

    document.querySelector('.close-icon-modal').addEventListener('click', () => toggleIconModal(false));
    typeFontBtn.addEventListener('click', () => toggleIconType('font'));
    typeImgBtn.addEventListener('click', () => toggleIconType('img'));
    
    document.getElementById('icon-class').addEventListener('input', (e) => {
        const icon = document.getElementById('preview-icon');
        icon.className = (e.target.value || 'fas fa-globe') + ' transition-all duration-300 drop-shadow-lg';
    });

    colorPicker.addEventListener('input', (e) => {
        const val = e.target.value;
        colorHex.value = val;
        previewIcon.style.color = val;
    });

    colorHex.addEventListener('input', (e) => {
        let val = e.target.value;
        if (!val.startsWith('#')) val = '#' + val;
        previewIcon.style.color = val;
        if (/^#[0-9A-F]{6}$/i.test(val)) {
            colorPicker.value = val;
        }
    });

    document.getElementById('icon-file').addEventListener('change', (e) => {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = (evt) => {
                const img = document.getElementById('preview-img');
                img.src = evt.target.result;
                updatePreviewVisibility(); 
            };
            reader.readAsDataURL(file);
        }
    });

    document.getElementById('btn-save-icon').addEventListener('click', () => {
        const name = document.getElementById('icon-name').value || '应用';
        const url = document.getElementById('icon-url').value || '#';
        const isFont = document.getElementById('type-font').classList.contains('active');
        let iconVal;
        
        if (isFont) {
            iconVal = document.getElementById('icon-class').value || 'fas fa-globe';
        } else {
            const img = document.getElementById('preview-img');
            if (img.src && img.src !== window.location.href && img.src !== '') {
                iconVal = img.src;
            } else {
                iconVal = document.getElementById('icon-class').value || 'fas fa-globe';
            }
        }

        const newApp = {
            id: currentEditId || Date.now(),
            name: name,
            url: url,
            type: (isFont || (!isFont && (!document.getElementById('preview-img').src || document.getElementById('preview-img').src === window.location.href))) ? 'font' : 'img',
            icon: iconVal,
            color: document.getElementById('icon-color-hex').value
        };

        if (currentEditId) {
            const idx = state.apps.findIndex(a => a.id === currentEditId);
            if (idx !== -1) state.apps[idx] = newApp;
        } else {
            state.apps.push(newApp);
        }
        saveState();
        renderGrid();
        toggleIconModal(false);
    });
}

function updatePreviewVisibility() {
    const isFont = document.getElementById('type-font').classList.contains('active');
    const previewIcon = document.getElementById('preview-icon');
    const previewImg = document.getElementById('preview-img');
    const hasImg = previewImg.src && previewImg.src !== window.location.href && previewImg.src !== '';

    if (isFont) {
        previewIcon.classList.remove('hidden');
        previewImg.classList.add('hidden');
    } else {
        if (hasImg) {
            previewIcon.classList.add('hidden');
            previewImg.classList.remove('hidden');
        } else {
            previewIcon.classList.remove('hidden');
            previewImg.classList.add('hidden');
        }
    }
}

function toggleIconType(type) {
    const fontGroup = document.getElementById('input-font-group');
    const imgGroup = document.getElementById('input-img-group');
    const btnFont = document.getElementById('type-font');
    const btnImg = document.getElementById('type-img');

    if (type === 'font') {
        btnFont.classList.add('active');
        btnImg.classList.remove('active');
        // 使用 CSS 类控制显隐，而不是 display:none
        fontGroup.classList.remove('hidden');
        imgGroup.classList.add('hidden');
    } else {
        btnImg.classList.add('active');
        btnFont.classList.remove('active');
        imgGroup.classList.remove('hidden');
        fontGroup.classList.add('hidden');
    }
    updatePreviewVisibility();
}

function openIconModal(editId = null) {
    currentEditId = editId;
    toggleIconModal(true);
    
    // Reset Fields
    document.getElementById('icon-name').value = '';
    document.getElementById('icon-url').value = '';
    document.getElementById('icon-class').value = '';
    
    // Reset Image Preview
    const previewImg = document.getElementById('preview-img');
    previewImg.src = ''; 
    previewImg.classList.add('hidden');
    
    // Reset Color
    const defaultColor = '#ffffff';
    document.getElementById('icon-color-picker').value = defaultColor;
    document.getElementById('icon-color-hex').value = defaultColor;
    document.getElementById('preview-icon').style.color = defaultColor;
    document.getElementById('preview-icon').classList.remove('hidden');

    if (editId) {
        const app = state.apps.find(a => a.id === editId);
        if (app) {
            document.getElementById('icon-name').value = app.name;
            document.getElementById('icon-url').value = app.url;
            
            if (app.type === 'font') {
                 document.getElementById('icon-class').value = app.icon;
            } else {
                 document.getElementById('icon-class').value = 'fas fa-globe';
                 previewImg.src = app.icon;
            }

            const color = app.color || defaultColor;
            document.getElementById('icon-color-picker').value = color;
            document.getElementById('icon-color-hex').value = color;
            document.getElementById('preview-icon').style.color = color;

            toggleIconType(app.type);
            
            document.getElementById('icon-class').dispatchEvent(new Event('input'));
        }
    } else {
        document.getElementById('icon-class').value = 'fas fa-globe';
        toggleIconType('font');
        document.getElementById('icon-class').dispatchEvent(new Event('input'));
    }
}

// ================= CONFIRM MODAL LOGIC =================
function showConfirmModal() {
    const wrapper = document.getElementById('confirm-modal-wrapper');
    const btnOk = document.getElementById('btn-confirm-ok');
    const btnCancel = document.getElementById('btn-confirm-cancel');

    // 显示弹窗
    wrapper.classList.add('active');

    // === 关键步骤：防止事件重复绑定 ===
    // 每次打开弹窗前，克隆按钮以清除之前的 EventListener
    // 否则点一次“删除”可能会触发多次
    const newBtnOk = btnOk.cloneNode(true);
    const newBtnCancel = btnCancel.cloneNode(true);
    btnOk.parentNode.replaceChild(newBtnOk, btnOk);
    btnCancel.parentNode.replaceChild(newBtnCancel, btnCancel);

    // 绑定取消按钮
    newBtnCancel.addEventListener('click', () => {
        wrapper.classList.remove('active');
    });

    // 绑定确认删除按钮
    newBtnOk.addEventListener('click', () => {
        if (ctxTargetId !== null) {
            // 执行删除逻辑
            state.apps = state.apps.filter(a => a.id !== ctxTargetId);
            saveState();
            renderGrid();
        }
        wrapper.classList.remove('active');
    });
}